@* Output Routines.
In order to demonstrate our program, we now add to
|Triangulation| a function |show()|, which draws (in the special case
|dmax==2|) the simplicial complex into a {\em LEDA}-|window|.
Running through the list |all_simplices| we draw each simplex.
For each simplex we draw its vertices and for each vertex
of a simplex we draw the edges connecting it to the other vertices
of the simplex.
Clearly we do not draw the anti-origin and the edges incident to it.
Thus the |for|-loop which steps through all vertices starts with
$v = 0$ if $S$ is bounded
(i.e., |S->vertices[0] != anti_origin|)
and with $v = 1$ if $S$ is unbounded
(i.e., |S->vertices[0] == anti_origin|).\\
Furthermore, we draw every point that we have inserted so far onto the
screen (there may be many points that are not vertices of any
simplex). We do this by running through the list |coordinates|.

@<Member f...@>+=
void Triangulation::show(window &W)
{
  // We first draw every simplex
  Simplex *S;
  forall(S, all_simplices) {
    for (int v= (S->vertices[0] == anti_origin ? 1 : 0);
	 v<=dcur; v++) { // for each vertex except the anti-origin
      d_rat_point x = coordinates.contents(S->vertices[v]);
      point a = convert(x);
      for (int e=v+1; e<=dcur; e++) { // draw undrawn edges incident to vertex 
	d_rat_point y = coordinates.contents(S->vertices[e]);
        point b = convert(y);
	// draw the edges of unbounded simplices as thick lines 
	if (S->vertices[0] == anti_origin) 
	  W.set_line_width(3);
	else 
	  W.set_line_width(1);
	W.draw_segment(a,b);
      }
    }
  }
  // Now we draw every point 
  d_rat_point x;

  forall(x,coordinates) {
    point a = convert(x);
    W.draw_point(a);
    }
}

@
|print_all()| prints information about all simplices to |stdout|.
This was useful for debugging.
The information of a single simplex is printed by the function
|print()|.
@<Member fu...@>+=
void Triangulation::print_all()
{
  Simplex *S;

  cout << "\n dcur " << dcur << " origin_simplex: ";
  if (origin_simplex!=nil)
    cout << origin_simplex->sim_nr;
  else
    cout << "none";
  cout << " quasi_center: " << quasi_center << endl;
  forall(S, all_simplices) {
    print(S);
  }
}



@ Here is a short function that prints the data of a simplex.
@<Member f...@>+=
void Triangulation::print(Simplex *S)
{
  list_item p;

  cout <<"\n[" << S->sim_nr << "]-----------------------------------------------------\n";

  for(int i=0;i<=dcur;i++) {
    if (S->vertices[i]==anti_origin)
      cout << "[xx] anti-origin";
    else 
      cout << '[' << order_nr[S->vertices[i]] << ']'
	   << coordinates.contents(S->vertices[i]);
    cout << " [";
    if (S->neighbors[i])
      cout << S->neighbors[i]->sim_nr << "] <-> [" << S->opposite_vertices[i];
    else
      cout << "*";
    cout << "] ";
    if ((S->vertices[0]!=anti_origin || i==0) && dcur > 0) {
      cout << ";  facet: " << S->facets[i];
      cout << "; valid_equations: " << S->valid_equations[i];
    }
   cout << endl;
  }
  cout << "Points: ";
  forall(p,S->points)
    cout << '[' << order_nr[p] << ']' << coordinates.contents(p) << ' ';
  cout << endl;
  cout.flush();
}

@
This is a function that puts the triangulation to a stream by simply
writing its defining points to it. That is done in a way that it can be
directly used as input for another Triangulation.
@<Friend functions of class...@>=

ostream& operator<<(ostream &o,Triangulation& T)
{
d_rat_point v;
int i;

  o << T.dmax << endl << T.coordinates.size() << endl;
  forall(v,T.coordinates)
   {
    for(i = 0; i<= T.dmax; i++)
      o << v.ccoord(i);
    o << endl;
   }
  return o;
}

@
This function merges the current triangulation with the Triangulation
in the input stream. The exspected format is the same as in the main
function.
@<Friend functions of class...@>+=

istream &operator>>(istream &i,Triangulation &T)
{
int dim;

i >> dim;
if (dim!=T.dmax)
  error_handler(20,"chull: different dimensions in stream"); // cannot merge
i >> dim; // ignore number of points

d_rat_point v(dim);
while(!i.eof())
 {
  i >> v;
  T.insert(v);
 }
return i;
}

@ 
We should have a function that outputs all outer points of the triangulation,
i.e. those located in unbounded simplices. This is useful for constructing a hull
describing the same set as the original hull but with a minimal amount of points
We print each point only once, so we test if it is already printed.

@<Member fun...@>+=
void Triangulation::print_extremes(ostream &o)
{
Simplex *sim;
list_item p;
int i;
h_array<list_item,bool> printed(false);
list<list_item> points;

forall(sim,all_simplices)
  if (sim->vertices[0]==anti_origin)
    for(i=1;i<=dcur;i++) // the zeroth point is the |anti_origin|
      points.append(sim->vertices[i]);

o << dmax << endl << points.size() << endl;
forall(p,points)
 {
  if (!printed[p]) {
    for ( i=0; i<= dmax; i++)
      o << coordinates[p].ccoord(i); 
    o  << endl;
    printed[p]=true;
  }
 }
}
