@* The Deletion Procedure.
\label{del_proc}
We now come to the deletion procedure. First we repeat the overview from
section \ref{del_overview}.

\deloversec

\medskip
We may run into six cases when we delete a point $x$.
\begin{itemize}
\item $x$ is not a point of the current hull, so we are done.
\item $x$ is the only point of a triangulation. We set the
triangulation to an initial state just as if no point was ever added.
\item $x$ lies in the interior of some simplex. We delete $x$ from the
list of inner points of this simplex.
\item Deleting $x$ reduces the dimension of the current hull. We have
to reverse the process for inserting a dimension jump (section
\ref{dimjmp}).
\item  Deleting $x$ leaves a "hole" in the hull to be filled with new
simplices induced by points inserted after $x$.
\item $x$ is a dimension jump, but there is another point prevending
us from reducing the dimension of the hull. We search this point,
declare it as new dimension jump and can go on almost exactly as in
the previous case.
\end{itemize}

The first three cases are rather trivial.
We use a |dictionary co_index| to look up the position (|list_item|) of
a point in the |list coordinates|. If we find none, |x| is not
in the hull.
Otherwise we get the |list_item item_x| of |x| from the |dictionary|.
We will use |item_x| throughout this function as it has more
information assigned to it than |x| and it is unique. |x| by itself is not
always unique, as there may be many points with the
same coordinates. By taking the last one of all possible
|list_item|s assigned to points with the same coordinates as |x|
we will remove an inner point if |x| was inserted more than
once, improving our running time a  bit this way.
When we found |x| in the current hull, we test if it is the
only point and reset if triangulation if it is so.
Next we test wether |x| is an inner point by simply looking at
it's |position| value.
If it is a vertex we handle the remaing cases in section
\ref{non-triv cases}.

@<Member f...@>+=
void Triangulation::del(const d_rat_point &x)
{
dic_item dic_x;

/* find |x| in the defining point set. By taking the last
inserted point, we find inner points when the same point is
inserted more often, making deletion easier this way. */

if ((dic_x = co_index.lookup(x)) == nil)
  return; // |x| not in the hull $\Rightarrow$ nothing to do

list_item item_x = co_index.inf(dic_x)->Pop();

if (co_index.inf(dic_x)->empty()) {
  delete co_index.inf(dic_x);
  co_index.del_item(dic_x);
}

if (coordinates.length()==1) // |x| is the last point in the hull
 {
  @<clean up triangulation@>@;
  return;
 }

if ((dcur == 0) || (position[item_x] != nil)) // |x| is not a vertex
 {
  @<delete inner point@>@;
  return;
 }

@<handle non-triv...@>@;
}

@ Here we set our triangulation to an initial state as it was before the first point
was inserted. We free all memory we allocated dynamically,
clear the lists of points and simplices, and set |dcur|,
|origin_simplex|, |simplex|, and |position| to their initial state.

@<clean up...@>=
{
Simplex *sim;

forall(sim,all_simplices)
  delete sim;
all_simplices.clear();
origin_simplex=nil;
coordinates.clear();
dcur=-1; @/
simplex = h_array<list_item, Simplex *> (nil); @/
position = h_array<list_item, list_item> (nil); @/
order_nr = h_array<list_item, int> (-1); @/
}

@ Here we delete an inner point of a simplex. This simplex is directly
determined by the pointer |simplex[item_x]|. In the same way the position
of |x| in the |list points| of the simplex is determined directly by the
|position| h\_array. So it can be deleted in (exspected) constant time.

@<delete inner point@>=
 {
  Simplex *sim;

  sim=simplex[item_x];
  sim->points.del_item(position[item_x]);
  position[item_x] = nil;
  simplex[item_x] = nil;
  order_nr[item_x] = -1;
  coordinates.del_item(item_x);
 }

@
\label{non-triv cases}
We need some additional information about the triangulation when we delete
a vertex.

\begin{zitat}{\cite{BMS}, p.\ 5}
We store some additional information in the insertion process\footnote{s. section
\ref{find_visible_facets} for actual implementation.}.
Each point $x_k$ is stored in exactly one simplex in $\Delta(\pi_k)$ that
contains $x_k$ in its closure\footnote{This only holds for non-vertices.}.
The insertion algorithm gives us such
a simplex. Furthermore we store the set of dimension jumps in a list\footnote{
The |vertices| of the |origin_simplex| are such a list.}.
For a vertex $x_l$ of $\overline{\Delta}(\pi)$ let $S(x_l)$ denote the
set of simplices with vertex $x_l$ and let $S_k(x_l)$ be the set of simplices
in $S(x_l)$ whose peak index is at most $k$. Furthermore let $R(x_l)$ denote
the set of points stored in the simplices in $S(x_l)$ and let $P(x_l)$
be the set of vertices that are opposite to $x_l$ ($x$ and $y$ are {\it
opposite} if there is a facet $F$ of $\overline{\Delta}$%
\footnote{IMHO should be $\Delta$, see source code for the difference}
 such that $S(F \cup x)$
and $S(F \cup y)$ are Simplices of $\overline{\Delta}$).
\end{zitat}

So we introduce the variables |Sx| to hold the information of $S(x)$, |Rx|
holding $R(x)$, and |Px| holding $P(x)$. The only difference is that $x$ is
not contained in |Rx| as it should according to the citation above. We can
omit it because we know this fact implicitely.
We also want to know if |x| is a dimension jump (i.e. a vertex of the
|origin_simplex|), so we use |is_dj| for that. It may have three values:
0 when |x| is not a dimension jump, 1 when |x| is a dimension jump and the
new dimension jump lies on the same halfspace as |x|, and 2 when |x| is a
dimension jump and the new dimension jump lies on the other halfspace as
|x|.
Finally, |facet| denotes the index of the facet of a simplex in |Sx|
opposite to |x|. You could also say that it is the index of |item_x| in the
|vertices|-array of a simplex in |Sx|.

Our case distinction of the last three cases is based on the following:

\begin{zitat}{\cite{BMS}, p.\ 6}
If $x_i$ was a dimension jump we have either $dim(R\setminus\{x_i\}) =
\dimc R - 1$
or we get a new dimension jump, say $x_j$. Deletion of $x_i$ reduces
dimension if all other points lie in $\aff(DJ\setminus\{x_i\})$. If $P(x_i) = \emptyset
\ $and$ \ \aff(R(x_i)\setminus\{x_i\}) = \aff(DJ\setminus\{x_i\})$ then $x_i$ reduces dimension.
\end{zitat}
Otherwise we have to find a new dimension jump and can go on as
if we delete a normal vertex.
The test is performed in a loop. Putting each point to be tested in a plane
equation takes less time then setting up an array and then calling
|is_containted_in_affine_hull()| (for |d_rat_point|s), which solves a system of linear
equations.

@<handle non-triv cases@>=
{
int is_dj=0;
list<Simplex *> Sx;
// list of pointers to simplices containing |x|
h_array<Simplex *,int> facet(-1); // facet[Sx]=facet towards x
list<list_item> Rx;
// pointers into list of points that are in simplices in Sx
list<list_item> Px;   // pointer into list of points opposite to x

@<set up variables@>@;

if ((is_dj!=0)&&(dcur>0))
 {
  if (Px.empty()) {
    bool hulls_equal=true;
    list_item p;

    forall(p,Rx)
      if (sees_facet(origin_simplex,facet[origin_simplex],
        	coordinates[p]) !=  0)
	hulls_equal=false;
    if (hulls_equal)
     {
      @<reduce dimension@>@;
      return;
     }
   }
  @<get new dimension jump@>@;
 }
@<delete non-dimjmp@>@;
}

@ \label{set up sets}
First we compute |Sx| by a recursiv function. Then we set |Px| to the list
of all points that are opposite to |x|. Remember that we consider $x$ and $y$
as opposite if there is a facet $F$ of $\Delta$ such that $S(F \cup x)$ and
$S(F \cup y)$ are simplices of $\Delta$. We say $\Delta$ in contrast to
$\overline{\Delta}$ as we want the |anti_origin| not to be in |Px|.

The computation of |Rx| is a bit more difficult. 
\begin{zitat}{\cite{CMS}, p. 12}
To determine $R(x)$, check first wether $x$ is a vertex of the simplices pointed to
by $x$. If not, $x$ is removed and we are done. If so, construct the set $R(x)$ by 
inspection of all simplices incident to $x$\footnote{These are the simplices in
|Sx|.}. This takes time proportional to $d$ times $\mid R(x) \mid$ plus the number of
removed simplices. Sorting the points in $R(x)$ by the time of insertion takes time
$O(\min \{ n, \mid R(x) \mid\log\log n \} )$, where the former bound is obtained by
bucketsort and the latter bound comes from the use of bounded ordered dictionaries.
\end{zitat}

Since |x| may be a vertex
of more than one simplex we have to exclude doubles. Additionally we want
to insert the inner points of simplices in |Sx|. So we use a {\sc Leda}
|h_array not_in_Rx| of |bool| to filter the doubles out.
Now we sort |Rx| according to the
insertion time (stored in |order_nr|) using {\em bucket sort}.
Our |sort|-function is an interface to the |bucket_sort()|-function for
{\sc Leda} |list|s. We need this for our goal to make the triangulation
look as if |x| was never added, so we have to preserve the insertion order.


@<set up variables@>=
{
int i;
list_item p;
Simplex *sim,*neighbor;
h_array<list_item, bool> not_in_Rx(true);
h_array<list_item, bool> not_in_Px(true);

is_dj=collect_Sx(simplex[item_x],item_x,Sx,facet);
clear_visited_marks(simplex[item_x]);
forall(sim,Sx)
 {
  neighbor=sim->neighbors[facet[sim]];
  p=neighbor->vertices[sim->opposite_vertices[facet[sim]]];
/* This if-clause is the difference between
$\overline{\Delta}$ and $\Delta$. */
  if ((neighbor->vertices[0]!=anti_origin) && (not_in_Px[p]==true)) {
    Px.append(neighbor->vertices[sim->opposite_vertices[facet[sim]]]);
    not_in_Px[p]=false;
  }

  for(i=0;i<=dcur;i++)
   {
    p=sim->vertices[i];
    if ((p!=anti_origin) && (p!=item_x) && (not_in_Rx[p]==true))
     {
      Rx.append(p);
      not_in_Rx[p]=false;
     }
   }
  forall(p,sim->points) {
    Rx.append(p);
  }
 }
sort(Rx);
}

@
This function collects  the simplices that belong to |Sx| and sets
up the |h_array facet| for the facets towards |x| of these simplices.
It returns 0 if |x| if not a dimension jump and 1 otherwise.
It is intended to be called with |simplex[item_x]| as initial |S|
and assumes that
|position[item_x] == nil|, i.e. that |x| is a vertex and not an
inner point.


@<Further mem...@>+=
int collect_Sx(Simplex *S,list_item item_x, @|
     list<Simplex *> &Sx, h_array<Simplex *,int> &facet);

@
We first mark the simplex |S| as visited. Then we test if our current
simplex is the |origin_simplex|. If so it is inserted at the top of the
list for later convinence, and a flag is set to indicate that |x| is a
dimension jump. All other simplices are simply appended. Then we determine
the |facet[S]| as the facet with the same index as |x| in the
|vertices|-array of |S|. We now recursivly visit all
neighbors of |S| that are not visited yet. We omit the one opposite to |x|
because it cannot contain |x| as a vertex. The return
values of the recursive calls are or'ed.

@<Member fun...@>+=
int Triangulation::collect_Sx(Simplex *S, list_item item_x, @|
     list<Simplex *> &Sx, h_array<Simplex *,int> &facet)
{
int i;
int is_dj=0;

S->visited=true;
if (S==origin_simplex)
 {
  is_dj=1;
  Sx.push(S); // insert the |origin_simplex| first
 }
else
  Sx.append(S); // all other simplices as they appear
for(i=0;i<=dcur;i++) {
  if (S->vertices[i]==item_x)
    facet[S]=i;
  else
    if (S->neighbors[i]->visited==false)
      is_dj|=collect_Sx(S->neighbors[i],item_x,Sx,facet);
}
return is_dj;
}

@
Here we reduce the dimension of a triangulation. This is easier as one
might think.\\
\begin{zitat}{\cite{BMS}, p.\ 6}
We simply remove all unbounded simplices that do not have $x_i$ in their
vertex set and remove $x_i$ from the vertex set of the remaining simplices.
\end{zitat}\\
Furthermore, other things to do are only for purposes of the internal
housekeeping. (Mostly remove all references to simplices in |Sx| from other simplices.)
Special care is needed to maintain the neighborhood relation intact when
reordering the arrays.
\begin{figure}[htb]
\begin{center}
\input{dj.latex}
\end{center}
\caption{We have this}
\end{figure}

\begin{figure}[htb]
\begin{center}
\input{dim1.latex}
\end{center}
\caption{We want this}
\end{figure}

This is the reversal of the process described in Section \ref{func_dimjmp}.
You might recognize the figures.

@<reduce dimension@>=
{
Simplex *sim,*neighbor;
int j;

while(!Sx.empty())
 {
  sim=Sx.pop();
  if (sim->vertices[0]!=anti_origin) {
    neighbor=sim->neighbors[facet[sim]];
  // these are to delete, as they were added with |x|
    all_simplices.del_item(neighbor->this_item);
    delete neighbor;
  }
  for(j=facet[sim];j<dcur;j++) // delete |x| from the vertex set
   {
    sim->vertices[j]=sim->vertices[j+1];
    sim->neighbors[j]=sim->neighbors[j+1];
    sim->opposite_vertices[j]=sim->opposite_vertices[j+1];
   }
  for(j=0;j<dcur;j++) {
    // adjust the neighborhood relation
    if (facet[sim->neighbors[j]]<sim->opposite_vertices[j])
      sim->opposite_vertices[j]--;
    // here we invalid the plane equations
    sim->valid_equations[j]=-1;
    // and we keep our simplex-pointer valid
    simplex[sim->vertices[j]]=sim;
  }
  // restore default state of some arrays
  sim->vertices[dcur]=nil;
  sim->opposite_vertices[dcur]=-1;
  sim->neighbors[dcur]=nil;
  sim->valid_equations[dcur]=-1;
 }
dcur--;  // some care for the internal management
quasi_center-=x;
position[item_x]=nil;
simplex[item_x] = nil;
order_nr[item_x] = -1;
coordinates.del_item(item_x);
}

@ Here we handle the deletion of a dimension jump that does not reduce
the dimension of the current hull because there is another point that
can serve as dimension jump. We have to find it, to reconstruct the
new |origin_simplex| and so on. A bit of theory as base for our
algorithm:\\
\begin{zitat}{\cite[p.\ 6]{BMS}}
By Lemma 2.1 dimension jumps can be moved to the front of the insertion order. Hence we
have \begin{displaymath}
\Delta(\pi \setminus \{x_i\}) = \Delta(\sigma\pi\setminus(\{x_i\}\cup DJ))
\end{displaymath} if $x_i \notin DJ$ and \begin{displaymath}
\Delta(\pi\setminus\{x_i\}) = \Delta(\sigma ` \pi\setminus DJ)
\end{displaymath}
if $x_i \in DJ$, where $\sigma$ and $\sigma `$ are arbitrary permutations of $DJ$ and $DJ
\setminus \{x_i\}$ resp. Hence we can assume $DJ \subset R_i$ wlog. If $x_i \in DJ$ and
$x_i$ is a new dimension jump we have by Lemma 2.2 \begin{displaymath}
\Delta(\pi\setminus\{x_i\}) = \Delta(\sigma ` \pi_{i-1}x_j\pi \setminus (R_i \cup DJ \cup
\{x_j\})).
\end{displaymath}
So we can reinsert $x_j$ first, thereby making sure that all dimension jumps are already
inserted when we reinsert the remaining points. The last paragraph descibes reinsertion
of a dimension jump.
\par [\ldots] \par
Let us finally consider reinsertion of a new dimension jump $x_j$. Since a new dimension
jump is reinserted first we can assume $j = i + 1$ wlog.
\end{zitat} \par
So we do (or better prepare everything so that is is done in section
\ref{del non-dj} here).
By making |x| the ``peak'' of the |origin_simplex| we can avoid a special
treatment.

@<get new dim...@>=
{
list_item new_dj; // pointer in |coordinates|
list_item p;
int f,h;
Simplex *sim;
hyperplane v;

@<find new dimjump@>@;
f=facet[origin_simplex];
p=origin_simplex->vertices[0];
origin_simplex->vertices[0]=item_x;
origin_simplex->vertices[f]=p;
sim=origin_simplex->neighbors[0];
origin_simplex->neighbors[0]=origin_simplex->neighbors[f];
origin_simplex->neighbors[f]=sim;
h=origin_simplex->opposite_vertices[0];
origin_simplex->opposite_vertices[0]=origin_simplex->opposite_vertices[f];
origin_simplex->opposite_vertices[f]=h;
v=origin_simplex->facets[0];
origin_simplex->facets[0]=origin_simplex->facets[f];
origin_simplex->facets[f]=v;
h=origin_simplex->valid_equations[0];
origin_simplex->valid_equations[0]=origin_simplex->valid_equations[f];
origin_simplex->valid_equations[f]=h;
facet[origin_simplex]=0;
if (sees_facet(origin_simplex,facet[origin_simplex],
      coordinates[new_dj]) < 0)
// if |x| and |new_dj| lie in different halfspace
  @<take care for new origin\_simplex@>@;
// adjust our center point
quasi_center-=x;
quasi_center+=coordinates[new_dj];
// the |new_dj| replaces the old also with its |order_nr|
order_nr[new_dj]=order_nr[item_x];
// continue as non-dimjump (section \ref{del non-dj})
}

@ Finding another dimension jump is described as follows.\\
 \begin{zitat}{\cite[p.\ 6]{BMS}}
 Otherwise we get a new dimension
jump $x_j$ where \begin{displaymath} j = min\{k;x_k
\notin \aff(DJ\setminus\{x_i\})\}. \end{displaymath}
\end{zitat}
We have to build a list of all |possible_points|. It consists of |Rx|
and |Px|. They cannot be simply concatenated because of the
implementation of {\sc Leda}'s $list.conc()$-function, which changes
both lists involved. So we have to copy them before we concatenate them.
Now we sort our |possible_points| according to their insertition order, so
we can take the first point in the list that matches the condition.

@<find new dimjump@>=
{
list<list_item> possible_points;
list<list_item> h;

possible_points=Rx;
h=Px;
possible_points.conc(h);
sort(possible_points);
 // note that |x| is {\sl not} in |possible_points|
 /* find the first point not in the affine hull of the remaining dimjmps*/
do {
  new_dj=possible_points.pop();
  }
while (sees_facet(origin_simplex,facet[origin_simplex],
	coordinates[new_dj]) == 0);
}

@ \begin{zitat}{\cite[p.\ 6]{BMS}}
If $x_i$ and $x_{i+1}$ are on
the same side of $\aff(DJ \setminus \{x_i\})$, then we take the simplices in $S_i(x_i)$,
replace $x_i$ by $x_{i+1}$ and add them to $\overline{\Delta}$. $B_{i+1}$ is the
collection of facets on the boundary of the union of these simplices that are not opposite
to $x_{i+1}$.
\end{zitat} \par
We do so in section \ref{del non-dj}. The only thing to make sure is
that the new |origin_simplex| is the first simplex built. Therefore
we inserted it to the top of list |Sx| (s. section \ref{set up sets}).
This guaranties the first simplex
built to fill the hole to be our new |origin_simplex|, with |new_dj|
as peak. So it remains nothing to do for now. Setting the new
|origin_simplex| can only be done when it is built. (s. section
\ref{special treatment})

@ The process for this case is described as follows:
\begin{zitat}{\cite[p.\ 6]{BMS}}
If $x_i$ and $x_{i+1}$ are on different sides of $\aff(DJ \setminus \{x_i\})$, no simplices
are added.
\end{zitat}
First we mark this case by setting |new_dj| to 2. Now we determine the new
|orgigin_simplex| as the simplex that shares the facet opposite to |x| with the old
|origin_simplex|. We also set the |facet| value for the new simplex to its correct
value. Then we assign the new |origin_simplex|.
After the |origin_simplex| is
handled this way, the new facets are collected in
section \ref{initialize newfacets}, so we do not have to do
anything more here.

@<take care for...@>=
{
Simplex *sim;

is_dj=2;
sim=origin_simplex->neighbors[facet[origin_simplex]];
facet[sim]=origin_simplex->opposite_vertices[facet[origin_simplex]];
origin_simplex=sim;
}

