@* Finding $x$-visible Hull Facets.
For finding the $x$-visible hull facets, we implement three
\label{search_methods}search methods.
The first method, the {\em visibility-search-method}, visits all simplices
with $x$-visible base facet using depth first search starting in the
origin simplex.
It is implemented in the function |visibility_search()|.

The second method is a {\em modified-visibility-search-method}.
The difference is that if it has once reached an outer simplex, it
restricts its search space to unbounded simplices.  It uses the
function |search_to_outside()|, which is similar to
|visibility_search()| except that it stops when it has reached an
unbounded simplex.
It returns a pointer to the unbounded simplex that it has
reached or |nil| if $x$ lies in the interior of the hull. If it has
reached an outer simplex, all unbounded $x$-visible simplices are
collected using the function |collect_outer_simplices()|.

The third method is the {\em segment-walking-method}. This method walks
\linebreak
through the simplices which are intersected by
a ray $\ray{Ox}$ from a point $O$ in the origin simplex to $x$.
It returns a pointer to the simplex it has reached (even if
this is a bounded simplex).
The unbounded $x$-visible simplices are also collected using the
function |collect_outer_simplices()|.

The visibility search method and the function
|collect_outer_simplices()| mark visited simplices as visited using
the |visited| variable.
We unmark them using the function |clear_visited_marks()|.

The roof function for these is |find_visible_facets()|. It switches
to the appropriate function und does the cleanup.

@<Further mem...@>+=
void visibility_search(Simplex* S, const d_rat_point &x);
Simplex* search_to_outside(Simplex *S, const d_rat_point &x);
Simplex* segment_walk(const d_rat_point &x);
void collect_visible_simplices(Simplex *S, const d_rat_point &x);
void clear_visited_marks(Simplex* S);
void find_visible_facets(const d_rat_point &x);

list<Simplex *> visible_simplices; // result of |find_visible_facets()|
Simplex *inner_simplex; // where |x| is located if |visible_simplices| is empty

@ The variable |method| (which can be changed interactively by the
user) switches between the several search methods.
\label{find_visible_facets}

@<Member func...@>+=
void Triangulation::find_visible_facets(const d_rat_point &x)
{
Simplex *last_simplex;
  // the simplex in which modified visibility search 
  // and segment walking have stopped
switch(method) {
case VISIBILITY:
  visibility_search(origin_simplex, x);
    // generates list of unbounded simplices with |x|-visible base facet
  clear_visited_marks(origin_simplex);
  if ((visible_simplices.empty())&&(inner_simplex==nil))
    inner_simplex=origin_simplex;
  break;
case MODIFIED_VISIBILITY:
  last_simplex=search_to_outside(origin_simplex,x);
  if(last_simplex!=nil) // if |x| is not an interior point
     collect_visible_simplices(last_simplex, x);
      // generates list of unbounded simplices with |x|-visible base facet
  else
    if (inner_simplex==nil) // |x| is interior but no simplex found
      inner_simplex=origin_simplex; // so its the origin simplex
  clear_visited_marks(origin_simplex);
  break;
case SEGMENT_WALK:@;
default:
  last_simplex=segment_walk(x);
  if(last_simplex->vertices[0] == anti_origin) { // if |x| is not an interior point
    collect_visible_simplices(last_simplex, x);
      // generates list of unbounded simplices with |x|-visible base facet
    clear_visited_marks(last_simplex);
  }
  else  // |x| lies in the interior
    inner_simplex=last_simplex;
  break;
}
}

@ How we can implement |visibility_search()| is described in
Section \ref{search}: starting at the origin simplex, we
visit any unvisited neighbor of a visited simplex that has an |x|-visible base
facet. Note that by this rule, we do not have to test the origin simplex
(which by definition has indeed no base facet).
The |class Triangulation| has a member
list |visible_simplices|, in which we store the outer simplices
seeing |x|. The function |visibility_search()| is recursive and gets as
arguments a reference to the vector $x$ and a pointer
|*S| to the simplex to be visited. 
\label{visibility search}

@<Member f...@>+=
void Triangulation::visibility_search(Simplex *S,@|
          const d_rat_point &x)
{
  searched_simplices++; // only for statistical reasons
  S->visited=true; // we have visited |S| and never come back \ldots
  for(int i=0;i<=dcur;i++) {
    Simplex *T=S->neighbors[i];  // for all neighbors |T| of |S|
    if(!T->visited) {
        // if the |i|-th neighbor has not been visited yet

      if (sees_facet(T,0,x) > 0) {
	// if |x| sees the base facet of the |i|-th neighbor

	if(T->vertices[0]==anti_origin)
          // if the |i|-th neighbor is an outer simplex
          visible_simplices.push(T);
          // we have found a visible simplex and store it
	else 
	  // test if |x| lies within |T|
	  {
	  bool in=true;
	  int j;
	  for(j=1;j<=dcur;j++)
	    if (sees_facet(T,j,x) < 0)
	      in=false;
          if (in)
	    inner_simplex=T;
          }
        visibility_search(T,x);
	// do the recursive search
      }
    }
  }
}


@ Here is the first part of the possibly faster modified visibility
search method: search from the origin simplex to the outside, then
search on the outer facets recursively with depth first search.  If
|x| is an outer point, that means it is contained in one of the outer
simplices, the function returns a pointer to the first outer simplex
that is found. If |x| is an inner point, the function returns |nil|.
When we say ``possibly faster'', we have in mind that the searching to
the outside (which is nothing but depth first search) will take
exactly the same way as the normal visibility search if $x$ is an
interior point, so the time we have spent is unfortunately the same.
It might only be faster if $x$ lies not in the current hull.

@<Member f...@>+=
Simplex* Triangulation::search_to_outside(Simplex *S, @|
       const d_rat_point &x)
{
  searched_simplices++; // only for statistical reasons
  S->visited=true; // we have visited |S| and never come back \ldots
  for(int i=0;i<=dcur;i++) {
    Simplex *T=S->neighbors[i];  // for all neighbors |T| of |S|
    if(!T->visited)
        // if the |i|-th neighbor has not been visited yet

      if (sees_facet(T,0,x) > 0) {
	// if |x| sees the base facet of the |i|-th neighbor

	if(T->vertices[0]==anti_origin)
          // if the |i|-th neighbor is an outer simplex
          return T; // we have found to the outside
	else
	 {
	  bool in=true; // test if $x$ sees the all facets
	  int j;

	  for (j=1;j<=dcur;j++)
	    if (sees_facet(T,j,x) < 0)
	      in=false;
          if (in)
	    inner_simplex=T;
         }
	Simplex *result=search_to_outside(T, x);
	if(result!=nil)
	  return result;
      }
  }
  return nil;
}

@
Now we collect all outer simplices which are visible from |x|. The
collection process starts from an outer simplex |S|.
@<Member f...@>+=
void Triangulation::collect_visible_simplices(Simplex* S, @|
           const d_rat_point &x)
{
  searched_simplices++; // only for statistical reasons
  S->visited=true; // we have visited |S| and never come back...
  visible_simplices.push(S); // store |S| as a visible simplex

  for(int i=0;i<=dcur;i++) {
    Simplex *T=S->neighbors[i];  // for all neighbors |T| of |S|
    if(!T->visited && T->vertices[0]==anti_origin) 
        // if the |i|-th neighbor has not been visited yet
        // and is an outer simplex
      if (sees_facet(T,0,x) > 0)
	// if |x| sees the base facet of the |i|-th neighbor
	collect_visible_simplices(T,x); // do the recursive collecting
      }
}


@ After a visibility search, we always must clear the |visited|-bits of
the visited simplices. This is done by the recursive function
|clear_visited_marks()|.
It is very similar to the function
|visibility_search()|. When we start this function, we also call it
with the origin simplex as its argument.
@<Member f...@>+=
void Triangulation::clear_visited_marks(Simplex *S)
{
  S->visited=false; // clear the |visited|-bit
  for(int i=0;i<=dcur;i++) // for all neighbors of |S|
    if(S->neighbors[i]->visited) // if the |i|-th neighbor has been visited
	clear_visited_marks(S->neighbors[i]); 
                // clear its bit recursively
}



@
The following function implements the segment walk method to find
the simplex containing the point $x$.
Let $O$ denote the ``origin'', i.e., any point in the origin
simplex; we can take |quasi_center/(dcur+1)| for $O$.
The strategy is very simple: we start at the origin simplex and
walk along the ray $\ray{Ox}$ through the simplices intersected
by this ray until we reach the simplex containing $x$.
In order to guarantee that the ray $\ray{Ox}$ passes only
through the interior of facets we perturb the point $O$. The
perturbation scheme is similiar to the well known perturbation
method for the simplex algorithm. Let $\epsilon$ be a positiv
infinitesimal, let $\Epsilon$ denote the point $(\epsilon,
\epsilon^2, \ldots , \epsilon^d)$ and let $O^\epsilon = O +
\Epsilon$.

The function |lambda_cmp()| (s. section \ref{lambda_cmp-sec})
computes the order in which the ray $\ray{Ox}$ intersects various
hyperplanes. Only here the perturbation plays a role.
We now develop the mathematics underlying this subroutine. 

Let $h$ be a hyperplane. A hyperplane is the zero-set of an
affine function $h(x) = \sum\limits_{0\leq i < d}h_i x_i + h_d$,
where $x = (x_0, \ldots , x_{d-1})$ is a point given by its
cartesian coordinates. Let $\overline{h}(x) = \sum\limits_{0\leq i <
d} h_i x_i$. Then $h(x) = \overline{h}(x) + h_d$,
$\overline{h}$ is linear, and 
$\overline{h}(x + y) = \overline{h}(x) +
\overline{h}(y)$, $\overline{h}(\lambda x) = 
\lambda\overline{h}(x)$, 
$h(x - y) = h(x) - \overline{h}(y)$, and
$h(x) - h(y) = \overline{h}(x) - \overline{h}(y)$.

The points on the ray $\ray{Ox}$ satisfy the equation
$$r(\lambda) = O^\epsilon + \lambda(x - O^\epsilon).$$
The parameter value $\lambda$ for which the ray $r$ intersects
the hyperplane $h$ is determined by the equation $0 =
h(r(\lambda)) = h(O^\epsilon) + \overline{h}(\lambda(x -
O^\epsilon)) = h(O^\epsilon) + \lambda(\overline{h}(x) -
\overline{h}(O^\epsilon)) = h(O^\epsilon) + \lambda(h(x) -
h(O^\epsilon))$. Thus 
\begin{equation}\label{lambda_h} \lambda_h = -
\frac{h(O^\epsilon)}{h(x) - h(O^\epsilon)}.\end{equation}
The perturbation method guaranties that the denominator is
non-zero. Since $O^\epsilon = O + \Epsilon$ we have $h(x) -
h(O^\epsilon) = h(x) - h(O) - \overline{h}(\Epsilon) = h(x) -
h(O) - \allowbreak \epsilon \cdot \sum\limits_{0 \leq i < d} h_i
\epsilon^i$. We conclude that 
\begin{equation}\label{sign_hx-hO}\sign (h(x) - h (O^\epsilon))
= \left\{ \begin{array}{l@@{\mbox{ , if }}l}
\sign(h(x) - h(O)) & h(x) - h(O) \not= 0 \\
-\sign(h_i) & \rule{0pt}{5ex}\makebox[10em][l]{\parbox{13em}{$h(x) - h(O) = 0$,\\
and $i$ is minimal with $h_i \not= 0$}}
\end{array}\right.\end{equation}
and analogous
\begin{equation}\label{sign_hOeps}\sign (h(O^\epsilon))
= \left\{ \begin{array}{l@@{\mbox{ , if }}l}
\sign(h(O)) & h(O) \not= 0 \\
\sign(h_i) & \rule{0pt}{5ex}\parbox{13em}{$h(O) = 0$, \\
and $i$ is minimal with $h_i \not= 0$}
\end{array}\right.\end{equation}
Note that at least one of the coefficients $h_0, \ldots ,
h_{d-1}$ is non-zero. Hence $h(x) - h(O^\epsilon)$ and $h(O^\epsilon)$
are always non-zero.

Let $g$ be another hyperplane and let $\lambda_g = -
g(O^\epsilon) / (g(x) - g(O^\epsilon))$ be the parameter value
for the intersection of $\ray{Ox}$ with $g$. This ray
intersects $g$ before $h$ iff $\lambda_g < \lambda_h$. We have
\addtolength{\jot}{2ex}% s.u.
\begin{eqnarray}
\label{denom}\lambda_g < \lambda_h & \iff & \frac{-
g(O^\epsilon)}{g(x) - g(O^\epsilon)} < \frac{-
h(O^\epsilon)}{h(x) - h(O^\epsilon)} \\
 & \iff & g(O^\epsilon)\cdot
(h(x)-h(O^\epsilon)) >^\sigma h(O^\epsilon)\cdot (g(x)-
g(O^\epsilon))\nonumber
\end{eqnarray}
where $>^\sigma$ indicates $>$ iff the signs of the two
denominators are equal and denotes $<$ otherwise. Using
$h(O^\epsilon) = h(O) + \overline{h}(\Epsilon)$ and simplifying
we obtain\hfill\mbox{}
\makebox[\textwidth][r]{
\begin{minipage}{1.04\textwidth}
\vspace{-0.3ex}
\begin{eqnarray}
\lambda_g < \lambda_h & \Longleftrightarrow & g(O)h(x) -
h(O)g(x) + \overline{g}(\Epsilon)h(x) -
\overline{h}(\Epsilon)g(x) >^\sigma 0 \nonumber\\
\label{E} & \Longleftrightarrow & \underbrace{g(O)h(x) - h(O)g(x) +
\epsilon\cdot\sum_{0\leq i < d} \epsilon^i (g_i h(x) - h_i
g(x))}_{\displaystyle E} >^\sigma 0
\end{eqnarray}
\vspace{0.3ex}
\end{minipage}
}
\addtolength{\jot}{-2ex}% s.o.
Under what circumstances is $\lambda_g = \lambda_h$ possible?
If $\lambda_g = \lambda_h$ then the expression $E$ must be
zero. This is only possible if either $h(x) = g(x) = 0$, i.e.,
$x$ lies on $h$ and on $g$, or if $h = g$.\footnote{Since $g(x) \not=
0$ implies $h_i = g_i h(x)/g(x)$ for $0\leq i <d$ and hence
$\overline{h}(y) = h(x)/g(x) \overline{g}(y)$ for all $y$.
From $0 = g(O)h(x) - h(O)g(x)$ we conclude further that
$0 = \overline{g}(O)h(x)+g_d h(x) - \overline{h}(O)g(x) -
h_d g(x) = \overline{g}(O) h(x) + g_d h(x) - (h(x)/g(x))
\overline{g}(O) g(x)- h_d g(x) = g_d h(x) - h_d g(x)$ and
therefore $h_d = g_d h(x)/g(x)$.}

The variable |in| tells us the number of the facet
through which we have entered current simplex. It changes,
when we walk from simplex to simplex.
When we start our walk, there is no facet through which we have entered
the current simplex (which is the origin simplex).
This is indicated by setting the variable |in| to $-1$
at the beginning of |segment_walk()|.
We stop our walk, if we have found the simplex containing $x$ (this
might be an unbounded simplex, of course).
We use two arrays |fx| and |fO| to store the values of the
plane equations at $x$ and $O$ which we will need several times.
We must not forget that these values have to be divided by |x[0]| or
|O[0]| respectivly (the common denominator of the homogeneous
coordinates) to get the correct value.

According to \cite{KM} the denominator is always positive, so
it does not change the result of {\sc Leda}'s |sign|-function
for |integer|s, which we use for a quick comparison against 0.

@<Member f...@>+=
Simplex* Triangulation::segment_walk(const d_rat_point &x)
{ Simplex *S = origin_simplex; // we start at the origin simplex
  bool x_in_S=false; // indicates whether we have found the simplex containing |x|
  int in = -1; // entry facet of the origin simplex
  int i; // for treating every facet of |S|
  d_rat_point O=quasi_center/(dcur+1); // our center point
  array<integer> fx(0,dcur); // $h_i(x) * $|x[0]|
  array<integer> fO(0,dcur); // $h_i(O) * $|O[0]|

  while(!x_in_S) {
    searched_simplices++; // only for statistical reasons
    @<Compute the arrays |fx| and |fO| and test whether $x \in S$@>@;
    if (!x_in_S) {
      @<Go to the next Simplex on the ray $\ray{Ox}$@>@; 
    }
  }
  return S;
}

@
The computation of the arrays is easily done by calling
|S->facet[i].value_at()|. At the same time, we can
test whether |x| lies in the current simplex.

@<Compute the arrays...@>=
{
x_in_S = true; // remains true until we find a facet which doesn't see $x$
if (S->vertices[0] != anti_origin) // otherwise we have reached the outside
 {
  for (i = 0; i <= dcur; i++) {
    compute_plane(S,i); // just in case we need it the first time
  // remember to divide these values by the denominator of the point
    fx[i]=S->facets[i].value_at(x);
    fO[i]=S->facets[i].value_at(O);
    if (sign(fx[i]) < 0) // see manual
      x_in_S = false;
  }
 }
}

@
For the comparison of two $\lambda$'s, we use the function
|lambda_cmp()| defined in section~\ref{lambda_cmp-sec}.
This function contains the actual perturbation method.
A call |lambda_cmp(S,Od,xd,gx,gO,g,hx,hO,h)| returns true iff
$\lambda_g < \lambda_h$.
We also need a function |lambda_negative()| (s. section \ref{lambda_negative-sec})
which tells us wether $\lambda_h$ is negative or not.

@<Further mem...@>+=
bool lambda_cmp(Simplex* S, const integer &Od, const integer &xd, @|
                const integer &gx, const integer &gO, int g, @|
                const integer &hx, const integer &hO, int h);
bool lambda_negative(Simplex *S,const integer &Od, const integer &xd, @|
                const integer &nx,const integer &nO,int i);

@ Now we describe how to find the facet with number |out| through
which we will leave the current simplex.
Basically we have to find a facet |out| such that for all facets $i \ne $|out| $\lambda_{out} <
\lambda_i$. However there is an additional condition to meet to go
only forward on the ray.
If we are in the |origin_simplex|, |in == -1| and we
entered the simplex from nowhere. Therefore we additionally have to
test if $\lambda$ is positive.
If we are not in the |origin_simplex|, our $\lambda_{out}$ has to be
greater then $\lambda_{in}$, because otherwise the ray intersected facet |out|
prior and we are looking for the next intersection and not for the
last. To put it into formulas, facet $i$ becomes the new candidate for
the exit facet |out|, iff
$$ \lambda_{in} < \lambda_i < \lambda_{out} $$
where $\lambda_{in}$ is assumed to be $0$ when |in == -1|.
The index of a facet is the index of the vertex opposite to it.
Hence we set |in| to |S->opposite_vertices[out]|.

@<Go to the next Simp...@>=
int start = 0; // prior to this facet we don't need to compare
int out; // our hypothesis for desired facet

/* this loop terminates because x doesn't lie in |S| when we come here
and |dcur| is at least 1 */
while ((start == in) || // don't compare with ourselves
       ((in == -1) && lambda_negative(S,O[0],x[0],fx[start],fO[start],start)) || 
	// when in the origin simplex we must not start with a negative lambda
       ((in != -1) && lambda_cmp(S,O[0],x[0],fx[start],fO[start],start,fx[in],fO[in],in)))
       // otherwise we must not start with $\lambda_{start} < \lambda_{in}$
    start++;  // move one ahead

out = start++;
for (i = start; i <= dcur; i++) // compare it to all others
{
  if ((i != in) && // we don't go back ....
      (i != out)) // don't compare with ourselves
    if ((lambda_cmp(S,O[0],x[0],fx[i],fO[i],i,fx[out],fO[out],out)) &&@|
        // the basic comparison ($\lambda_i < \lambda_{out}$)
        ((in != -1) || (!lambda_negative(S,O[0],x[0],fx[i],fO[i],i))) && @|
        // additionally for the |origin_simplex|
        ((in == -1) || (lambda_cmp(S,O[0],x[0],fx[in],fO[in],in,fx[i],fO[i],i))))
        // additionally for all other simplices ($\lambda_{in} < \lambda_i$)
      out=i;
}
in=S->opposite_vertices[out];
S=S->neighbors[out];

@
It remains to describe how we decide whether
\label{lambda_cmp-sec}$\lambda_g < \lambda_h$.
As we have already mentioned we perturb\footnote{This perturbation
method was proposed by Kurt Mehlhorn}
$O$ to get $O^\epsilon := O + \Epsilon$, where
$\Epsilon = (\epsilon, \epsilon^2, \ldots, \epsilon^{dcur})$
for some sufficiently small $\epsilon > 0$. Thus, if we consider
$\epsilon$ small enough, the perturbated ray will not go through any
vertex or any other intersection of the hyperplanes of the
triangulation. Therefore, the facets which are intersected by this
ray are totally linearly ordered. In equation \ref{E} we developed a
condition to test. 

The expression $E$ contains two parts: one independ of $\Epsilon$ and
one depending on $\Epsilon$. When the first part is $\not= 0$, it
determines the sign of $E$ because $\epsilon$ is small enough.
If it is zero, the sign is determined by the first factor next to
$\epsilon^i$ that is $\not= 0$.

Now let's have a look at the implementation.
In {\sc Leda} a |d_rat_point| $x = (x_0, \ldots , x_{d-1})$ is
represented in homogeneous coordinates $(G_0, \ldots ,G_d)$
with $G_d > 0$ and $x_i = G_i/G_d$. Let 
$$H(x) = \sum\limits_{0
\leq i \leq d} h_i G_i = G_d \cdot \left( \sum\limits_{0 \leq
i < d} \frac{h_i G_i}{G_d}  + h_d \right) = G_d \cdot h(x)$$
We draw some simple conclusions:
\addtolength{\jot}{1.5ex}% s.u.
\begin{eqnarray}
& g(O) h(x) - h(O) g(x) =
\frac{G(O) H(x)}{o_d G_d} - \frac{H(O) G(x)}{o_d G_d} & \nonumber\\
\label{c_gOhx-hOgx}\Longrightarrow & \sign(g(O)h(x) - h(O)g(x))
= \sign(G(O)H(X) - H(O)G(x)) & \\
& g_ih(x) - h_ig(x) = \frac{g_i H(x)}{G_d}
- \frac{h_i G(x)}{G_d} & \nonumber\\
\label{c_gihx-higx}\Longrightarrow & \sign(g_ih(x)-h_ig(x)) =
\sign(g_iH(x)-h_iG(x)) &
\end{eqnarray}
\addtolength{\jot}{-1.5ex}%s.o.
Finally observe that $H(p) =$ |h.value_at(p)| for a |d_rat_point p|
\allowbreak and a \linebreak |hyperplane h|.
In conjunction with equation \ref{E} it is now easy to decide whether 
$\lambda_g < \lambda_h$.

|lambda_cmp()| gets as parameter the simplex |S| the facets belong to,
$o_d$ which is |Od| in the current implementation, $G_d =$|xd|,
$G(x)=$|gx|, $G(O)=$|gO|, the number |g| of the facet supporting the
hyperplane |g|, and the analogous parts for the hyperplane |h|.

We first have to decide upon $>^\sigma$.
For this reason, we use a variable |sigma| 
which is true iff $>^\sigma$ is $>$.

@<Member f...@>+=
bool Triangulation::lambda_cmp(Simplex *S, @|
           const integer &Od, const integer &xd,@|
           const integer &gx, const integer &gO, int g,@|
	   const integer &hx, const integer &hO, int h)
{ 
  bool sigma; 
  int diffsign;
  bool diffgr0;
  int i;

  @<Decide whether $>^\sigma$ is $<$ or $>$@>@;

  i=1;
  // First we test the parts which are not depending on $\Epsilon$.
  diffsign=sign(gO*hx-hO*gx); 
  // left part of $E$ in \ref{E} using \ref{c_gOhx-hOgx}
  while(diffsign==0) {
  /* the comparison depends on the factor of $\Epsilon$ when we enter
     this while-loop */
    diffsign=sign(S->facets[g][i]*hx-S->facets[h][i]*gx);
    // right part of $E$ ($\sum\ldots$) in \ref{E} using \ref{c_gihx-higx}
    i++;
  }
  diffgr0 = (diffsign > 0);
  return !(diffgr0^sigma);
}

@ Here we examine the direction of $>^\sigma$. We use another simple
conclusion 
\addtolength{\jot}{1.5ex}%s.u.
\begin{eqnarray}
& h(x) - h(O) = \frac{H(x)}{G_d} - \frac{H(O)}{o_d} = \frac{o_d
H(x) - G_d H(O)}{G_d o_d} & \nonumber\\
\label{c_hx-hO}\Longrightarrow & \sign(h(x) - h(O)) = 
\sign(o_dH(x) - G_d H(O)) & 
\end{eqnarray}
\addtolength{\jot}{-1.5ex}% s.o.
and equation \ref{sign_hx-hO} to inspect the sign of the 
denominators in equation \ref{denom}.

@<Decide whether $...@>=
{
int lsign,rsign;
int i;

// first compute the sign of the first denominator
i=1;
lsign=sign(Od*gx-xd*gO);
while (lsign==0) // |lsign| depends on $\Epsilon$
  lsign=-sign(S->facets[g][i++]);
// now the second
i=1;
rsign=sign(Od*hx-xd*hO);
while (rsign==0) // |rsign| depends on $\Epsilon$
  rsign=-sign(S->facets[h][i++]);
sigma=(lsign==rsign);
}

@
To decide upon the sign of a $\lambda_h$, we use the function
|lambda_negative()| which is similar to |lambda_cmp()| and returns
\label{lambda_negative-sec}true iff $\lambda_h < 0$.
We have to decide upon the sign of the numerator and denominator of 
equation \ref{lambda_h}.
Using our developed formulas \ref{sign_hx-hO} and \ref{sign_hOeps}
and our conclusion \ref{c_hx-hO}
this thing does not become a problem either. We do not forget the
'$-$'-sign in front of the right expression in equation \ref{lambda_h}.

@<Member f...@>+=
bool Triangulation::lambda_negative(Simplex *S, @|
           const integer &Od,const integer &xd, @|
           const integer &hx, const integer &hO, int h)
{ 
int zsign,nsign; // signs of numerator and denominator
int i;

// first the numerator
i=1;
zsign=sign(hO);
while (zsign==0)
  zsign=sign(S->facets[h][i++]);
// now for the denominator
i=1;
nsign=sign(Od*hx-xd*hO);
while (nsign==0)
  nsign=-sign(S->facets[h][i++]);
return (zsign==nsign);
}

